/** \file pfGrab.h
  */


/*!
 * \mainpage pfGrab DLL Documentation
 * \version 1.0
 * \author Photonfocus AG
 * \date 01/2010
 *
 * \section Intro   Introduction
 * In order to ease the acquisition of frames suitable to be used with 3rd party 
 * application we offer a range of frame grabber drivers along with our Pflib SDK
 * or alternative the PFConfig  library. Each frame grabber driver file is an 
 * ordinary shared library called DLL file in Windows. That exposes a known 
 * C interface in order to use a vendor-specific grabber SDK or API.
 *
 * There are examples available about how to use the pfGrab DLL.
 *
 *
 */

#ifndef PFGRAB_H
#define PFGRAB_H

#ifdef PFGRAB_EXPORTS
#ifdef __cplusplus
extern "C"
#define PFGRAB_API  extern "C" __declspec(dllexport)
#else
#define PFGRAB_API __declspec(dllexport)
#endif

#else
#ifdef __cplusplus
#define PFGRAB_API extern "C" __declspec(dllimport)
#else
#define PFGRAB_API __declspec(dllimport)
#endif
#endif

#include <stdio.h>
#include <string>

enum PFGRAB_ERROR
{
    /* No error. */
    PFGRAB_ERROR_OK = 0,

    /* Unknown error. */
    PFGRAB_ERROR_UNKNOWN = -1,

    /* Memory allocation. */
    PFGRAB_ERROR_MEMORY = -2,

    /* The file is not found or an invalid format. */
    PFGRAB_ERROR_INVALID_FILE = -5,

    /* The frame grabber has not being started to grab yet. */
    PFGRAB_ERROR_FRAME_GRABBER_NOT_STARTED = -8,

    /* The frame grabber has been stopped while waiting for a frame. */
    PFGRAB_ERROR_FRAME_GRABBER_STOPPED = -9,

    /* The area is invalid. */
    PFGRAB_ERROR_INVALID_AREA = -10,

    /* There was a timeout performing the operation. */
    PFGRAB_ERROR_TIMEOUT = -16,

    /* A frame grabber driver reported some error unknown to the PFGRAB API. */
    PFGRAB_ERROR_FRAME_GRABBER_UNKNOWN = -18,
};


using namespace std;

///
/// \brief Initializes the frame grabber driver. 
/// 
/// \param[in] The string parameter is a free-form string to create a new pfGrab object.
/// It will be used to specify the frame grabber's configuration file,
/// the port to use or even which ones of the system installed frame grabbers to use. 
/// This string is driver-specific and is documented for every frame grabber driver,
/// even if it gets ignored by the driver.
///
/// \return The return value is the pointer to the driverData. 
/// This pointer is later passed as a void pointer to all other functions
/// in order to operate on the correct data. If there was an error in the initialization 
/// of the frame grabber driver the return value is NULL.
///
PFGRAB_API void* pfGrab_InitFG (int port, const char *fgName, const char *param);

///
/// \brief Most frame grabbers, for performance reasons, can be configured with a 
/// region of interest (ROI) that specifies which part of the whole image we are 
/// interested and the grabber device will only transfer the data of that region.
/// Normally the region of interest is specified in the configuration file for the
/// frame grabber but some applications require change the size of the ROI.
///  If the new region of interest's size could be set successful the function returns 
/// PFGRAB_ERROR_OK=0. If the given width or height is not allowed the function corrects
/// the value and returns the new value. Keep in mind that the user is instructed to never
/// change the region of interest while grabbing images.
///
/// \param[in] driverData The pointer to the driverData.
/// \param[in] width Image width to set
/// \param[in] height Image height to set
///
/// \return PFGRAB_ERROR_OK if successful otherwise error code.
///
PFGRAB_API long pfGrab_SetSize (void *driverData, long *width, long *height);

///
/// \brief The function pfGrab_StartGrab instructs the frame grabber driver
/// to start grabbing new frames. This function initialize the FG callbacks,
/// queues, and any other data structure given by the grabber device's vendor.
///	
/// \param[in] driverData The pointer to the driverData.
///
/// \return Return PFGRAB_ERROR_OK if there was no error starting
/// to acquire new frames.
///
PFGRAB_API long pfGrab_StartGrab (void * driverData);

///
/// \brief A call to pfGrab_Wait means that pfGrab is waiting 
/// for the frame grabber driver to acquire a new frame. 
/// From the point of view of pfGrab a new frame is whatever 
/// the grabber device acquired since the last call to pfGrab_StartGrab 
/// or pfGrab_Wait. That means that if since the last call to wait
/// three new frames have been acquired then the next call to wait won't 
/// need to wait to acquire the next frame, it is already available. 
///
/// \param[in] driverData The pointer to the driverData.
/// \param[in] timeout The parameter expressed in milliseconds that sets
/// the maximum number of milliseconds to wait for the next frame. 
/// If the driver wasn't able to acquire a new frame in the specified 
/// number of milliseconds then it returns PFGRAB_ERROR_TIMEOUT.
/// Take into account that a negative number of milliseconds as timeout
/// means that the driver must ignore the timeout and just wait 
/// until there's a frame available.
///
/// \return Return PFGRAB_ERROR_OK if a new image is available.
///
PFGRAB_API long pfGrab_Wait (void * driverData, long timeout);


///
/// \brief The function pfGrab_GetImage is called by the application
/// when there was a successful call to pfGrab_Wait. The purposes 
/// of this function if to get the pointer to the frame object 
/// that was created while waiting for an acquisition.
///
/// \param[in] driverData The pointer to the driverData.
/// \param[in] buffer Passing an optional buffer address for copying
/// the image data to the destination of the buffer address. If the
/// buffer address is NULL, the parameter is ignored.
/// \param[in] size In the case a buffer address is passed, the
/// function copies  the number of bytes defined in size to the
/// buffer destination.
/// 
/// \return The pointer to the frame object that was created 
/// while waiting for an acquisition.
///
PFGRAB_API void* pfGrab_GetImage (void * driverData, void* buffer, long* bufferSize);


///
/// \brief The function pfGrab_StopGrab is called when the user is not 
/// interested in receiving more frames. The driver tells the hardware to 
/// stop acquiring frames and, if it kept a queue, to clean up the 
/// acquired frames that weren't returned to the user.
/// 
/// \param[in] driverData The pointer to the driverData.
///
/// \return Return PFGRAB_ERROR_OK if there was no error starting 
/// to acquire new frames.
///
PFGRAB_API long pfGrab_StopGrab(void *driverData);


///
/// \brief When pfGrab no longer needs to keep the driver around because
/// no one is using it anymore the driver must clean up its own resources. 
/// The application should call the release function when the driver
/// must clean up itself
///
/// \param[in] driverData The pointer to the driverData.
///
///
PFGRAB_API void pfGrab_ReleaseFG (void * driverData);

#endif
